﻿using System;
using Microsoft.SPOT;
using Microsoft.SPOT.Hardware;

using GT = Gadgeteer;
using GTM = Gadgeteer.Modules;
using GTI = Gadgeteer.Interfaces;

using Mountaineer.Netmf.Hardware;

namespace Gadgeteer.Modules.Mountaineer
{
    /// <summary>
    /// A ButtonForMountaineer module for Microsoft .NET Gadgeteer
    /// </summary>
    public class ButtonForMountaineer : GTM.Module
    {
        // Note: A constructor summary is auto-generated by the doc builder.
        /// <summary></summary>
        public ButtonForMountaineer()
        {            
            this.builtInButton = new InterruptPort(OnboardIO.Button, true, Port.ResistorMode.Disabled, Port.InterruptMode.InterruptEdgeBoth);
            this.builtInButton.OnInterrupt += new NativeEventHandler(this._port_OnInterrupt);
        }

        private void _port_OnInterrupt(uint data1, uint data2, DateTime time)
        {
            ButtonState value = data2 > 0 ? ButtonState.Pressed : ButtonState.Released;     // Active high!
            this.OnButtonEvent(this, value);
        }

        private InterruptPort builtInButton;

        /// <summary>
        /// Gets a value that indicates whether the button is pressed.
        /// </summary>
        public bool isButtonPressed
        {
            get
            {
                return this.builtInButton.Read();  // Active high!
            }
        }

        /// <summary>
        /// Represents the state of button of the <see cref="ButtonForMountaineer"/>.
        /// </summary>
        public enum ButtonState
        {
            /// <summary>
            /// The button is released.
            /// </summary>
            Released = 0,  // Active high!
            /// <summary>
            /// The button is pressed.
            /// </summary>
            Pressed = 1     // Active high!
        }

        /// <summary>
        /// Represents the delegate that is used to handle the <see cref="ButtonPressed"/>
        /// and <see cref="ButtonReleased"/> events.
        /// </summary>
        /// <param name="sender">The <see cref="ButtonForMountaineer"/> object that raised the event.</param>
        /// <param name="state">The state of the <see cref="ButtonForMountaineer"/></param>
        public delegate void ButtonEventHandler(ButtonForMountaineer sender, ButtonState state);

        /// <summary>
        /// Raised when the state of <see cref="ButtonForMountaineer"/> is pressed.
        /// </summary>
        /// <remarks>
        /// Implement this event handler and/or the <see cref="ButtonPressed"/> event handler
        /// when you want to provide an action associated with button events.
        /// The state of the button is passed to the <see cref="ButtonEventHandler"/> delegate,
        /// so you can use the same event handler for both button states.
        /// </remarks>
        public event ButtonEventHandler ButtonPressed;

        /// <summary>
        /// Raised when the state of <see cref="ButtonForMountaineer"/> is released.
        /// </summary>
        /// <remarks>
        /// Implement this event handler and/or the <see cref="ButtonReleased"/> event handler
        /// when you want to provide an action associated with button events.
        /// The state of the button is passed to the <see cref="ButtonEventHandler"/> delegate,
        /// you can use the same event handler for both button states.
        /// </remarks>
        public event ButtonEventHandler ButtonReleased;

        private ButtonEventHandler onButton;

        /// <summary>
        /// Raises the <see cref="ButtonPressed"/> or <see cref="ButtonReleased"/> event.
        /// </summary>
        /// <param name="sender">The <see cref="ButtonForMountaineer"/> that raised the event.</param>
        /// <param name="buttonState">The state of the button.</param>
        protected virtual void OnButtonEvent(ButtonForMountaineer sender, ButtonState buttonState)
        {
            if (this.onButton == null)
            {
                this.onButton = new ButtonEventHandler(this.OnButtonEvent);
            }

            if (buttonState == ButtonState.Pressed)
            {
                // Program.CheckAndInvoke helps event callers get onto the Dispatcher thread.  
                // If the event is null then it returns false.
                // If it is called while not on the Dispatcher thread, it returns false but also re-invokes this method on the Dispatcher.
                // If on the thread, it returns true so that the caller can execute the event.
                if (Program.CheckAndInvoke(ButtonPressed, this.onButton, sender, buttonState))
                {
                    this.ButtonPressed(sender, buttonState);
                }
            }
            else
            {
                if (Program.CheckAndInvoke(ButtonReleased, this.onButton, sender, buttonState))
                {
                    this.ButtonReleased(sender, buttonState);
                }
            }
        }
    }
}
