﻿using System;
using Microsoft.SPOT;
using Microsoft.SPOT.Hardware;

using GT = Gadgeteer;

using Mountaineer.Netmf.Hardware;
using Microsoft.SPOT.Hardware.AnalogOutput;
using Mountaineer.Gadgeteer.MainboardWithBuiltinButton;

namespace Mountaineer.Gadgeteer
{
    /// <summary>
    /// Support class for The Mountaineer Group MountaineerEth for Microsoft .NET Gadgeteer
    /// </summary>
    public class MountaineerEth : GT.Mainboard
    {
        // The mainboard constructor gets called before anything else in Gadgeteer (module constructors, etc), 
        // so it can set up fields in Gadgeteer.dll specifying socket types supported, etc.

        /// <summary>
        /// Instantiates a new MountaineerEth mainboard
        /// </summary>
        public MountaineerEth()
        {
            // uncomment the following if you support NativeI2CWriteRead for faster DaisyLink performance
            // otherwise, the DaisyLink I2C interface will be supported in Gadgeteer.dll in managed code.
            GT.Socket.SocketInterfaces.NativeI2CWriteReadDelegate nativeI2C = null; // new GT.Socket.SocketInterfaces.NativeI2CWriteReadDelegate(NativeI2CWriteRead);

            GT.Socket socket;

            // socket 1
            socket = GT.Socket.SocketInterfaces.CreateNumberedSocket(1);
            socket.SupportedTypes = new char[] { 'C', 'Y' };
            socket.CpuPins[3] = Socket1.Pin3;
            socket.CpuPins[4] = Socket1.Pin4;
            socket.CpuPins[5] = Socket1.Pin5;
            socket.CpuPins[6] = Socket1.Pin6;
            socket.CpuPins[7] = Socket1.Pin7;
            socket.CpuPins[8] = Socket1.Pin8;
            socket.CpuPins[9] = Socket1.Pin9;
            socket.NativeI2CWriteRead = nativeI2C;
            GT.Socket.SocketInterfaces.RegisterSocket(socket);

            // socket 2
            socket = GT.Socket.SocketInterfaces.CreateNumberedSocket(2);
            socket.SupportedTypes = new char[] { 'S', 'U', 'Y' };
            socket.CpuPins[3] = Socket2.Pin3;
            socket.CpuPins[4] = Socket2.Pin4;
            socket.CpuPins[5] = Socket2.Pin5;
            socket.CpuPins[6] = Socket2.Pin6;
            socket.CpuPins[7] = Socket2.Pin7;
            socket.CpuPins[8] = Socket2.Pin8;
            socket.CpuPins[9] = Socket2.Pin9;
            socket.NativeI2CWriteRead = nativeI2C;
            socket.SerialPortName = Socket2.SerialPortName;
            socket.SPIModule = SPI.SPI_module.SPI1;
            GT.Socket.SocketInterfaces.RegisterSocket(socket);

            // socket 3
            socket = GT.Socket.SocketInterfaces.CreateNumberedSocket(3);
            socket.SupportedTypes = new char[] { 'K', 'U', 'Y' };
            socket.CpuPins[3] = Socket3.Pin3;
            socket.CpuPins[4] = Socket3.Pin4;
            socket.CpuPins[5] = Socket3.Pin5;
            socket.CpuPins[6] = Socket3.Pin6;
            socket.CpuPins[7] = Socket3.Pin7;
            socket.CpuPins[8] = Socket3.Pin8;
            socket.CpuPins[9] = Socket3.Pin9;
            socket.NativeI2CWriteRead = nativeI2C;
            socket.SerialPortName = Socket3.SerialPortName;
            GT.Socket.SocketInterfaces.RegisterSocket(socket);

            // socket 4
            socket = GT.Socket.SocketInterfaces.CreateNumberedSocket(4);
            socket.SupportedTypes = new char[] { 'P', 'Y' };
            socket.CpuPins[3] = Socket4.Pin3;
            socket.CpuPins[4] = Socket4.Pin4;
            socket.CpuPins[5] = Socket4.Pin5;
            socket.CpuPins[6] = Socket4.Pin6;
            socket.CpuPins[7] = Socket4.Pin7;
            socket.CpuPins[8] = Socket4.Pin8;
            socket.CpuPins[9] = Socket4.Pin9;
            socket.NativeI2CWriteRead = nativeI2C;
            socket.PWM7 = Socket4.Pwm7;
            socket.PWM8 = Socket4.Pwm8;
            socket.PWM9 = Socket4.Pwm9;
            GT.Socket.SocketInterfaces.RegisterSocket(socket);

            // socket 5
            socket = GT.Socket.SocketInterfaces.CreateNumberedSocket(5);
            socket.SupportedTypes = new char[] { 'S', 'U', 'Y' };
            socket.CpuPins[3] = Socket5.Pin3;
            socket.CpuPins[4] = Socket5.Pin4;
            socket.CpuPins[5] = Socket5.Pin5;
            socket.CpuPins[6] = Socket5.Pin6;
            socket.CpuPins[7] = Socket5.Pin7;
            socket.CpuPins[8] = Socket5.Pin8;
            socket.CpuPins[9] = Socket5.Pin9;
            socket.NativeI2CWriteRead = nativeI2C;
            socket.SerialPortName = Socket5.SerialPortName;
            socket.SPIModule = SPI.SPI_module.SPI3;
            GT.Socket.SocketInterfaces.RegisterSocket(socket);

            // socket 6
            socket = GT.Socket.SocketInterfaces.CreateNumberedSocket(6);
            socket.SupportedTypes = new char[] { 'A', 'I', 'O', 'X' };
            socket.CpuPins[3] = Socket6.Pin3;
            socket.CpuPins[4] = Socket6.Pin4;
            socket.CpuPins[5] = Socket6.Pin5;
            socket.CpuPins[6] = Socket6.Pin6;
            // Pin 7 not connected on this socket, so it is left unspecified
            socket.CpuPins[8] = Socket6.Pin8;
            socket.CpuPins[9] = Socket6.Pin9;
            socket.NativeI2CWriteRead = nativeI2C;
            socket.AnalogOutput = new MountaineerEth_AnalogOut(Socket6.Pin5);
            GT.Socket.SocketInterfaces.SetAnalogInputFactors(socket, 3.3, 0, 12);
            socket.AnalogInput3 = Socket6.AnalogInput3;
            socket.AnalogInput4 = Socket6.AnalogInput4;
            socket.AnalogInput5 = Socket6.AnalogInput5;
            GT.Socket.SocketInterfaces.RegisterSocket(socket);

            // socket 7
            socket = GT.Socket.SocketInterfaces.CreateNumberedSocket(7);
            socket.SupportedTypes = new char[] { 'U', 'X' };
            socket.CpuPins[3] = Socket7.Pin3;
            socket.CpuPins[4] = Socket7.Pin4;
            socket.CpuPins[5] = Socket7.Pin5;
            socket.CpuPins[6] = Socket7.Pin6;
            // Pin 7 not connected on this socket, so it is left unspecified
            // Pin 8 not connected on this socket, so it is left unspecified
            // Pin 9 not connected on this socket, so it is left unspecified
            socket.NativeI2CWriteRead = nativeI2C;
            socket.SerialPortName = Socket7.SerialPortName;
            GT.Socket.SocketInterfaces.RegisterSocket(socket);

            // socket 8
            socket = GT.Socket.SocketInterfaces.CreateNumberedSocket(8);
            socket.SupportedTypes = new char[] { 'Z' };
            // Pin 3 not connected on this socket, so it is left unspecified
            // Pin 4 is connected, but not made available
            // Pin 5 is connected, but not made available
            // Pin 6 not connected on this socket, so it is left unspecified
            // Pin 7 not connected on this socket, so it is left unspecified
            // Pin 8 not connected on this socket, so it is left unspecified
            // Pin 9 is connected, but not made available
            socket.NativeI2CWriteRead = null;
            GT.Socket.SocketInterfaces.RegisterSocket(socket);
        }

        bool NativeI2CWriteRead(int SocketNumber, GT.Socket.Pin sda, GT.Socket.Pin scl, byte address, byte[] write, int writeOffset, int writeLen, byte[] read, int readOffset, int readLen, out int numWritten, out int numRead)
        {
            // implement this method if you support NativeI2CWriteRead for faster DaisyLink performance
            // otherwise, the DaisyLink I2C interface will be supported in Gadgeteer.dll in managed code. 
            numRead = 0;
            numWritten = 0;
            return false;
        }

        private static string[] sdVolumes = new string[] { "SD" };

        /// <summary>
        /// Allows mainboards to support storage device mounting/umounting.  This provides modules with a list of storage device volume names supported by the mainboard. 
        /// </summary>
        public override string[] GetStorageDeviceVolumeNames()
        {
            return sdVolumes;
        }

        /// <summary>
        /// Functionality provided by mainboard to mount storage devices, given the volume name of the storage device (see <see cref="GetStorageDeviceVolumeNames"/>).
        /// This should result in a <see cref="Microsoft.SPOT.IO.RemovableMedia.Insert"/> event if successful.
        /// </summary>
        public override bool MountStorageDevice(string volumeName)
        {
            // implement this if you support storage devices. This should result in a <see cref="Microsoft.SPOT.IO.RemovableMedia.Insert"/> event if successful and return true if the volumeName is supported.
            return volumeName == "SD";
        }

        /// <summary>
        /// Functionality provided by mainboard to ummount storage devices, given the volume name of the storage device (see <see cref="GetStorageDeviceVolumeNames"/>).
        /// This should result in a <see cref="Microsoft.SPOT.IO.RemovableMedia.Eject"/> event if successful.
        /// </summary>
        public override bool UnmountStorageDevice(string volumeName)
        {
            // implement this if you support storage devices. This should result in a <see cref="Microsoft.SPOT.IO.RemovableMedia.Eject"/> event if successful and return true if the volumeName is supported.
            return volumeName == "SD";
        }

        /// <summary>
        /// Changes the programming interafces to the one specified
        /// </summary>
        /// <param name="programmingInterface">The programming interface to use</param>
        public override void SetProgrammingMode(GT.Mainboard.ProgrammingInterface programmingInterface)
        {
            // Change the reflashing interface to the one specified, if possible.
            // This is an advanced API that we don't expect people to call much.
        }


        /// <summary>
        /// This sets the LCD configuration.  If the value GT.Mainboard.LCDConfiguration.HeadlessConfig (=null) is specified, no display support should be active.
        /// If a non-null value is specified but the property LCDControllerEnabled is false, the LCD controller should be disabled if present,
        /// though the Bitmap width/height for WPF should be modified to the Width and Height parameters.  This must reboot if the LCD configuration changes require a reboot.
        /// </summary>
        /// <param name="lcdConfig">The LCD Configuration</param>
        public override void SetLCDConfiguration(GT.Mainboard.LCDConfiguration lcdConfig)
        {
        }

        /// <summary>
        /// Configures rotation in the LCD controller. This must reboot if performing the LCD rotation requires a reboot.
        /// </summary>
        /// <param name="rotation">The LCD rotation to use</param>
        /// <returns>true if the rotation is supported</returns>
        public override bool SetLCDRotation(GT.Modules.Module.DisplayModule.LCDRotation rotation)
        {
            return false;
        }

        // change the below to the debug led pin on this mainboard
        private const Cpu.Pin DebugLedPin = OnboardIO.LedBlue;

        private Microsoft.SPOT.Hardware.OutputPort debugled = new OutputPort(DebugLedPin, false);
        /// <summary>
        /// Turns the debug LED on or off
        /// </summary>
        /// <param name="on">True if the debug LED should be on</param>
        public override void SetDebugLED(bool on)
        {
            debugled.Write(on);
        }

        /// <summary>
        /// This performs post-initialization tasks for the mainboard.  It is called by Gadgeteer.Program.Run and does not need to be called manually.
        /// </summary>
        public override void PostInit()
        {
            return;
        }

        /// <summary>
        /// The mainboard name, which is printed at startup in the debug window
        /// </summary>
        public override string MainboardName
        {
            get { return "The Mountaineer Group MountaineerEth"; }
        }

        /// <summary>
        /// The mainboard version, which is printed at startup in the debug window
        /// </summary>
        public override string MainboardVersion
        {
            get { return "1.0"; }
        }

    }



    // This example class can be used to implement Analog Out support, if present on this mainboard
    internal class MountaineerEth_AnalogOut : GT.Socket.SocketInterfaces.AnalogOutput
    {
        // Declare a mainboard-specific analog output interface here (set to null since it is inactive)
        private AnalogOutput aout = null;

        Cpu.Pin pin;
        readonly double _minVoltage = 0;
        readonly double _maxVoltage = 3.3;

        public MountaineerEth_AnalogOut(Cpu.Pin pin)
        {
            this.pin = pin;
        }

        public double MinOutputVoltage
        {
            get
            {
                return _minVoltage;
            }
        }

        public double MaxOutputVoltage
        {
            get
            {
                return _maxVoltage;
            }
        }

        public bool Active
        {
            get
            {
                return aout != null;
            }

            set
            {
                if (value == Active) return;
                if (value)
                {
                    // Instantiate a mainboard-specific analog output interface here
                    aout = new AnalogOutput(AnalogOutputChannel.ANALOG_OUTPUT_0, 12);
                }
                else
                {
                    // Stop the mainboard-specific analog output interface here
                    aout.Dispose();
                    aout = null;
                }
            }
        }

        public void SetVoltage(double voltage)
        {
            Active = true;

            // Check that voltage does not fall outside of mix/max range
            if (voltage < _minVoltage)
                throw new ArgumentOutOfRangeException("The minimum voltage of the analog output interface is 0.0V");

            if (voltage > _maxVoltage)
                throw new ArgumentOutOfRangeException("The maximum voltage of the analog output interface is 3.3V");

            // Use the mainboard-specific analog
            aout.Write(voltage / _maxVoltage);
        }
    }
}
